<?php

namespace App\Http\Controllers\SatuSehat;

use App\Http\Controllers\Controller;
use App\Http\Requests\OrganisasiRequest;
use App\Models\SatuSehat\Organization;
use App\Models\SatuSehat\OrganizationReference;
use App\Services\SatuSehat\OrganizationService;
use App\Services\SatuSehat\SendRequestService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use DataTables;

class OrganizationController extends Controller
{
    function index(Request $req)
    {
        $data['referensi'] = OrganizationReference::all();
        $data['all_organisasi'] = DB::table('smis_ss_organisasi')->select('*')->where('slug_klinik', env('slug_klinik'))->get();
        
        return view('satu_sehat.organisasi.index', $data);
    }

    function datatable(){
        $query = DB::table('smis_ss_organisasi as org1')->leftJoin('smis_ss_organisasi as org2', 'org1.part_of', 'org2.id_organisasi')->select('org1.*', 'org2.nama as bagian_dari')
        ->where('org1.slug_klinik', env('slug_klinik'));
        return DataTables::of($query)->make(true);
    }

    function select_organisasi(Request $req){
        $data = Organization::findOrFail($req->id);
        return response()->json($data);
    }

    function datatable_referensi(Request $req)
    {
        $query = DB::table('smis_ss_organization_reference')->select('*');
        return DataTables::of($query)->make(true);
    }

    function update_referensi(SendRequestService $sss, OrganizationService $os)
    {
        try {
            $xml = simplexml_load_string($sss->get('http://terminology.hl7.org/CodeSystem/organization-type')->getBody()->getContents());
            $data = $xml->text->div->table;
            if ($data) {
                $os->store_referensi($data);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Gagal tersambung ke satu sehat'
                ]);
            }
            return response()->json([
                'status' => true,
                'message' => 'Update data referensi organisasi berhasil'
            ]);
        } catch (\Throwable $th) {
            return response()->json([
                'status' => false,
                'message' => $th->getMessage()
            ]);
        }
    }

    function store(OrganisasiRequest $req, SendRequestService $sss, OrganizationService $os)
    {
        $data = $os->data_add_organisasi($req);
        try {
            $result = $sss->post(env('SATU_SEHAT_URL') . '/fhir-r4/v1/Organization', json_encode($data));

            if ($result->getStatusCode() == 401) {
                $autentikasi = json_decode($sss->auth());
                Session::put('token_satu_sehat', $autentikasi ? $autentikasi->access_token : null);
                return redirect()->back()->with('gagal', 'Token expired, silahkan coba kembali');
            }

            if ($result->getStatusCode() == 201) {
                $data = json_decode($result->getBody()->getContents());
                $os->store($req, $data);
                return redirect('satu_sehat/organization')->with('sukses', 'Tambah organisasi berhasil');
            } else {
                return redirect()->back()->with('gagal', 'Gagal tambah organisasi');
            }
        } catch (\Throwable $th) {
            return redirect()->back()->with('gagal', $th->getMessage());
        }
    }

    function update(OrganisasiRequest $req, SendRequestService $sss, OrganizationService $os)
    {
        $data = $os->data_edit_organisasi($req);
        try {
            $result = $sss->put(env('SATU_SEHAT_URL') . '/fhir-r4/v1/Organization/'.$req->id_organisasi, json_encode($data));

            if ($result->getStatusCode() == 401) {
                $autentikasi = json_decode($sss->auth());
                Session::put('token_satu_sehat', $autentikasi ? $autentikasi->access_token : null);
                return redirect()->back()->with('gagal', 'Token expired, silahkan coba kembali');
            }

            if ($result->getStatusCode() == 200) {
                $os->update($req);
                return redirect('satu_sehat/organization')->with('sukses', 'Ubah organisasi berhasil');
            } else {
                return redirect()->back()->with('gagal', 'Gagal ubah organisasi');
            }
        } catch (\Throwable $th) {
            return redirect()->back()->with('gagal', $th->getMessage());
        }
    }
}
