<?php

namespace App\Http\Controllers;

use App\Services\LaporanTindakanDetailDokterService;
use App\Services\SuratKeteranganSehatService;
use App\Services\SuratSakitService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use PDF;
use RealRashid\SweetAlert\Facades\Alert;
use Yajra\DataTables\DataTables;

class SuratSakitController extends Controller
{
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            $menu = json_decode(Auth::user()->menu);
            if (property_exists($menu, 'medical_record')) {
                $arr = (array) $menu->medical_record;
                if ($arr['surat_sakit'] == 0) {
                    return redirect('home');
                }
            } else {
                return redirect('home');
            }
            return $next($request);
        });
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, SuratSakitService $suratSakitService)
    {
        return view('surat_sakit.index', [
            'datas' => $suratSakitService->getSuratSakit($request->keyword)->paginate(10),
            'keyword' => $request->keyword ?? ""
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(LaporanTindakanDetailDokterService $laporanTindakanDetailDokterService, SuratSakitService $suratSakitService, SuratKeteranganSehatService $suratKeteranganSehatService)
    {
        return view('surat_sakit.add_edit', [
            'dokters' => $laporanTindakanDetailDokterService->getDokterName(),
            'diagnosas' => $suratSakitService->getDiagnosa(),
            'no_surat' => $suratSakitService->getNoSk(),
            'address' => $suratKeteranganSehatService->getAddress(),
            'tanggal' => $suratKeteranganSehatService->konversiTanggal(date('d-m-Y')),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request, SuratSakitService $suratSakitService)
    {
        $createData = $suratSakitService->store($request);
        if (is_int($createData)) {
            if ($request->action == "cetak") {
                return redirect(route('surat_sakit.download', $createData));
            } else {
                Alert::success("Sukses", "Data Berhasil Disimpan");
                return redirect(route('surat_sakit.index'));
            }
        }
        return redirect()->back()->withErrors($createData);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(
        $id,
        LaporanTindakanDetailDokterService $laporanTindakanDetailDokterService,
        SuratSakitService $suratSakitService,
        SuratKeteranganSehatService $suratKeteranganSehatService
    ) {
        $data_surat = $suratSakitService->getSuratSakitById($id);
        return view('surat_sakit.add_edit', [
            'tanggal' => $suratKeteranganSehatService->konversiTanggal(date('d-m-Y')),
            'no_surat' => $data_surat->no_surat,
            'dokters' => $laporanTindakanDetailDokterService->getDokterName(),
            'keyword' => $request->keyword ?? "",
            'diagnosas' => $suratSakitService->getDiagnosa(),
            'data_surat' => $data_surat,
            'address' => $suratKeteranganSehatService->getAddress()
        ]);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id, SuratSakitService $suratSakitService)
    {
        $deleteData = $suratSakitService->delete($id);
        if ($deleteData == "sukses") {
            Alert::success("Sukses", "Data Berhasil Dihapus");
            return redirect(route('surat_sakit.index'));
        }
        return redirect()->back()->withErrors($updateData);
    }

    public function diagnosa(SuratSakitService $suratSakitService)
    {
        $data = $suratSakitService->getDiagnosa();
        return DataTables::of($data)->addIndexColumn()->make(true);
    }

    public function download($id, SuratKeteranganSehatService $suratKeteranganSehatService, SuratSakitService $suratSakitService)
    {
        $alamat = $suratKeteranganSehatService->getAddress();
        $data_surat = $suratSakitService->getSuratSakitById($id);
        PDF::setOptions(['dpi' => 150, 'defaultFont' => 'sans-serif']);
        $pdf = PDF::loadView('surat_sakit.pdf', ['address' => $alamat, 'data_surat' => $data_surat])->setPaper('a4', 'portrait')->setWarnings(false);
        return $pdf->download('Surat Keterangan Sakit ' . $data_surat->nama . '.pdf');
    }
}
